from .paths import FluidLabPaths
from bpy.types import AddonPreferences
from ..addon.naming import FluidLabNaming
from bpy.props import BoolProperty, StringProperty, EnumProperty
from ..libs.functions.get_common_vars import get_common_vars


class FluidLabAddonPreferences(AddonPreferences):
    bl_idname = FluidLabPaths.ROOT_BASENAME

    # Tabs
    tabs_prefs: EnumProperty(
        items=(
            ('RENDERING',       "Rendering",        "", 0),
            ('EXPORT_IMPORT',   "Export/Import",    "", 1),
            ('DP',              "Dynamic Paint",    "", 2),
        ),
        default='RENDERING'
    )

    # Preview Rendering
    flipbooks_path: StringProperty(
        name="Preview Folder",
        default="",
        subtype='DIR_PATH'
    )
    flipbook_without_overlays: BoolProperty(
        default=True
    )
    use_flipbooks_in_subfolders: BoolProperty(
        default=True
    )
    only_for_img_sequences: BoolProperty(
        default=True
    )
    # Export/Import
    exports_path: StringProperty(
        name="Exports path",
        default="",
        subtype='DIR_PATH'
    )

    def dp_cache_path_update(self, context):
        
        # Cambio on the fly todos los image_output_path de todos los canvas:
        
        fluid_colliders = get_common_vars(context, get_fluid_colliders=True)
        all_items_colliders = fluid_colliders.get_all_items

        for collider_item in all_items_colliders:

            # Ahora los colliders pueden ser varios:
            colliders_obs = [c_ob.ob for c_ob in collider_item.colliders]
            for collider_ob in colliders_obs:

                md = collider_ob.modifiers.get(FluidLabNaming.DP_CANVAS_MOD)
                if not md:
                    continue
                
                canvas = md.canvas_settings
                if not canvas:
                    continue

                surface = canvas.canvas_surfaces.active
                if not surface:
                    continue

                surface.image_output_path = self.dp_cache_path

    dp_cache_path: StringProperty(
        name="Cache Folder",
        description="Cache Folder for Bakes of Dinamic Paint Canvas",
        default="",
        subtype='DIR_PATH',
        update=dp_cache_path_update
    )

    def draw(self, context):
        layout = self.layout

        layout.use_property_split = True
        layout.use_property_decorate = False

        main_col = layout.column(align=True)
        box0 = main_col

        tabs = main_col.row(align=True)
        tabs.use_property_split = False
        tabs.scale_y = 1.3
        tabs.prop(self, "tabs_prefs", expand=True)

        main_col.separator()
        main_col.separator()

        if self.tabs_prefs == 'RENDERING':
            prev_header = box0.box().row(align=True)
            prev = box0.box().column(align=True)
            prev_header.alignment = 'CENTER'
            prev_header.label(text="Preview Rendering")

            prev.prop(self, "flipbooks_path", text="Preview Output")
            prev.prop(self, "flipbook_without_overlays", text="Overlays Off when Flipbook Start")
            prev.prop(self, "use_flipbooks_in_subfolders", text="Use subfolders")

            if self.use_flipbooks_in_subfolders:
                prev.prop(self, "only_for_img_sequences", text="Only for img sequences")
        
        elif self.tabs_prefs == 'EXPORT_IMPORT':
            e_i_header = box0.box().row(align=True)
            e_i = box0.box().column(align=True)
            e_i_header.alignment = 'CENTER'
            e_i_header.label(text="Export/Import")
            
            e_i.prop(self, "exports_path", text="Exports Path")
        
        elif self.tabs_prefs == 'DP':

            dp_header = box0.box().row(align=True)
            dp = box0.box().column(align=True)
            dp_header.alignment = 'CENTER'
            dp_header.label(text="Dynamic Paint")
            
            dp.prop(self, "dp_cache_path", text="Cache Path")

